%  Description:
%   1. This is a simple MatLab demo to animate the process of convolution.
%      It is meant to help student to visualize how convolution works.
%  
%   2. When this script is run, two function f(t) and go(t) are convolved
%      and the output figure will show animated graphical convolution.
%
%   3. The functions "f" and "go" and their range of interval can be changed by
%      editing this script itself in line numbers around "48 to 60"
% 
%   4. Note:  For a better scaled plots of the functions f(t) and go(t1), 
%             it is recommended to set the functions such that their  
%             maximum value remains comparable. e.g one can use appropriate
%             scaling. Other function are also given 'commented out'
%
%             Interger values are recommended for the ranges
%  
%   5. The animation can be made faster or slower by chaning the value of
%      the pause function in the animation loop. (around line number 118)

%  Author: Andreas Floros
%  Digital Audio Technology Toolbox
%  Audio Group, WCL, University of Patras
%  Last Modified: 17/05/2005.

% color of axis constant
  axis_color= [0.5 0.5 0.5];

% sampling interval constant
  s_int = 0.1;

% range for function 'f(t)'
  t = [ -10:s_int:10 ];

% definition of function 'f(t)'
   f = 0.1*(t.^2);
%  f = 5*ones(1, length(t)); 
%  f = t;

% range for function 'go(t1)'
  t1 = [-7:s_int:7];

% definition of function 'go(t1)'
  go = -0.1*(t1.^2);
% go = 10*(t1.^.3);
% go = 5*ones(1, length(t1));

% convolve: note the multiplation by the sampling interval
  c = s_int * conv(f, go);

% Animation %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% flip 'go(t1)' for the graphical convolutions g = go(-t1)
  g = fliplr(go);
  tf = fliplr(-t1);

% slide range of 'g' to discard non-ovelapping areas with 'f' in the convolution
  tf = tf + ( min(t)-max(tf) );

% get the range of function 'c' which is the convolution of 'f(t)' and 'go(t1)'
  tc = [ tf t(2:end)];
  tc = tc+max(t1);

% start graphical output with three subplots
  a_fig = figure;
  set(a_fig, 'Name', 'Animated Convolution', 'unit', 'pixel', ...
             'Position', [300, 150, 550, 640]);

% plot f(t) and go(t1)  
  ax_1 = subplot(3,1,1);
  op = plot(t,f, 'b',  t1, go, 'r');
  hold on; grid on;
  set(ax_1, 'XColor', axis_color, 'YColor', axis_color, 'Color', 'w');
  xlim( [ ( min(t)-abs(max(tf)-min(tf)) - 1 ) ( max(t)+abs(max(tf)-min(tf)) + 1 ) ] );
  title('Graph of f(t) - blue   &  go(t1) - red ', 'Color', axis_color );

% initialize animation the plot of 'g' is slided over the plot of 'f'

% plot f in the subplot number 2
  ax_2 = subplot(3,1,2);
  p = plot(t, f);
  hold on; grid on;
  title('Graphical Convolution: f(t) and g = go(-t1)', 'Color', axis_color );
  
% plot g in the subplot number 2
  q = plot(tf, g, 'r');
  xlim( [ ( min(t)-abs(max(tf)-min(tf))-1 ) ( max(t)+abs(max(tf)-min(tf))+1 ) ] );
  u_ym = get(ax_2, 'ylim');

% plot two vertical lines to show the range of ovelapped area
  s_l = line( [min(t) min(t)], [u_ym(1) u_ym(2)], 'color', 'g'  );
  e_l = line( [min(t) min(t)], [u_ym(1) u_ym(2)], 'color', 'g'  );
  hold on; grid on;
  set(ax_2, 'XColor', axis_color, 'YColor', axis_color, 'Color', 'w');

% initialize the plot the convolution result 'c'
  ax_3 = subplot(3,1,3);
  r = plot(tc, c);
  grid on; hold on;
  set(ax_3, 'XColor', axis_color, 'YColor', axis_color);
  xlim( [ min(tc)-1 max(tc)+1 ] );
  title('Convolutional Product c(t)', 'Color', axis_color );

% animation block
  for i=1:length(tc)
    
    % control speed of animation minimum is 0, the lower the faster
      pause(0.05);
      
    % update the position of sliding function 'g', its handle is 'q'
      tf=tf+s_int;
      set(q,'EraseMode','xor');
      set(q,'XData',tf,'YData',g);

    % show overlapping regions
    
    % show a vetical line for a left boundary of overlapping region
      sx = min( max( tf(1), min(t) ), max(t) );  
      sx_a = [sx sx];
      set(s_l,'EraseMode','xor');
      set(s_l, 'XData', sx_a);

    % show a second vetical line for the right boundary of overlapping region
      ex = min( tf(end), max(t) );  
      ex_a = [ex ex];
      set(e_l,'EraseMode','xor');
      set(e_l, 'XData', ex_a);
   
    % update the plot of convolution result 'c', its handle is r
      set(r,'EraseMode','xor');
      set(r,'XData',tc(1:i),'YData',c(1:i) );
    
  end;





